/*
 * Copyright 2013 two forty four a.m. LLC <http://www.twofortyfouram.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * <http://www.apache.org/licenses/LICENSE-2.0>
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package com.jfsoftware.ledcontrol;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Color;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.View;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.SeekBar;
import android.widget.Switch;
import android.widget.TableLayout;

import eu.chainfire.libsuperuser.Shell;

/**
 * This is the "Edit" activity for a Locale Plug-in.
 * <p>
 * This Activity can be started in one of two states:
 * <ul>
 * <li>New plug-in instance: The Activity's Intent will not contain
 * {@link com.twofortyfouram.locale.Intent#EXTRA_BUNDLE}.</li>
 * <li>Old plug-in instance: The Activity's Intent will contain
 * {@link com.twofortyfouram.locale.Intent#EXTRA_BUNDLE} from a previously saved plug-in instance that the
 * user is editing.</li>
 * </ul>
 *
 * @see com.twofortyfouram.locale.Intent#ACTION_EDIT_SETTING
 * @see com.twofortyfouram.locale.Intent#EXTRA_BUNDLE
 */
public final class EditActivity extends AbstractPluginActivity
{
    private SeekBar seekBarRed;
    private SeekBar seekBarGreen;
    private SeekBar seekBarBlue;
    private Button btnColor;
    private EditText txtColor;
    private EditText txtDelay;
    private Switch switchOff;
    private static Boolean _sync;
    private Boolean _disableEvents = false;

    @Override
    protected void onCreate(final Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        BundleScrubber.scrub(getIntent());

        final Bundle localeBundle = getIntent().getBundleExtra(com.twofortyfouram.locale.Intent.EXTRA_BUNDLE);
        BundleScrubber.scrub(localeBundle);

        setContentView(R.layout.main);
        seekBarRed = (SeekBar)findViewById(R.id.seekBarRed);
        seekBarGreen = (SeekBar)findViewById(R.id.seekBarGreen);
        seekBarBlue = (SeekBar)findViewById(R.id.seekBarBlue);
        btnColor = (Button)findViewById(R.id.btnColor);
        txtColor = (EditText)findViewById(R.id.txtColor);
        TableLayout tableSync = (TableLayout)findViewById(R.id.tableSync);
        txtDelay = (EditText)findViewById(R.id.txtDelay);
        switchOff = (Switch)findViewById(R.id.switchOff);

        if(!Shell.SU.available()) {
            AlertDialog.Builder builder = new AlertDialog.Builder(this);
            builder.setTitle(getString(R.string.title_noroot));
            builder.setMessage(getString(R.string.message_noroot));
            builder.setPositiveButton(getString(R.string.button_ok), new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    dialog.cancel();
                }
            });
            builder.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    mIsCancelled = true;
                    finish();
                }
            });
            builder.create().show();
            return;
        }

        if (TaskerPlugin.Setting.hostSupportsSynchronousExecution(getIntent().getExtras())) {
            tableSync.setVisibility(View.VISIBLE);
            _sync = true;
        } else {
            tableSync.setVisibility(View.GONE);
            _sync = false;
        }

        Boolean isGood = false;
        if (null == savedInstanceState)
        {
            if (PluginBundleManager.isBundleValid(localeBundle))
            {
                int red = localeBundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_RED);
                int green = localeBundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_GREEN);
                int blue = localeBundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_BLUE);
                _disableEvents = true;
                seekBarRed.setProgress(red);
                seekBarGreen.setProgress(green);
                seekBarBlue.setProgress(blue);
                btnColor.setBackgroundColor(Color.argb(255, red, green, blue));
                txtColor.setText(toHex(red) + toHex(green) + toHex(blue));
                FireReceiver.setLight(red, green, blue, false);
                if(_sync) {
                    int delay = localeBundle.getInt(PluginBundleManager.BUNDLE_EXTRA_INT_DELAY);
                    if(delay == 0) {
                        txtDelay.setText("");
                    } else {
                        txtDelay.setText(Integer.toString(delay));
                    }
                    switchOff.setChecked(localeBundle.getBoolean(PluginBundleManager.BUNDLE_EXTRA_BOOL_OFF));
                }
                _disableEvents = false;
                isGood = true;
            }
        }
        if(!isGood) {
            _disableEvents = true;
            seekBarRed.setProgress(0);
            seekBarGreen.setProgress(0);
            seekBarBlue.setProgress(0);
            txtColor.setText("000000");
            btnColor.setBackgroundColor(Color.argb(255, 0, 0, 0));
            FireReceiver.setLight(0, 0, 0, false);
            if(_sync) {
                txtDelay.setText("");
                switchOff.setChecked(true);
            }
            _disableEvents = false;
        }
        SeekBar.OnSeekBarChangeListener seekBarChangeListener = new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
                if(_disableEvents) return;
                final int red = seekBarRed.getProgress();
                final int green = seekBarGreen.getProgress();
                final int blue = seekBarBlue.getProgress();
                _disableEvents = true;
                btnColor.setBackgroundColor(Color.argb(255, red, green, blue));
                txtColor.setText(toHex(red) + toHex(green) + toHex(blue));
                _disableEvents = false;
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {

            }

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                if(_disableEvents) return;
                final int red = seekBarRed.getProgress();
                final int green = seekBarGreen.getProgress();
                final int blue = seekBarBlue.getProgress();
                FireReceiver.setLight(red, green, blue, false);
            }
        };
        seekBarRed.setOnSeekBarChangeListener(seekBarChangeListener);
        seekBarGreen.setOnSeekBarChangeListener(seekBarChangeListener);
        seekBarBlue.setOnSeekBarChangeListener(seekBarChangeListener);
        txtColor.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                if (_disableEvents) return;
                String input = txtColor.getText().toString();
                if (input.length() != 6) return;
                int red = 0;
                int green = 0;
                int blue = 0;
                try {
                    red = Integer.parseInt(input.substring(0, 1), 16);
                } catch (NumberFormatException ignored) {
                }
                try {
                    green = Integer.parseInt(input.substring(2, 3), 16);
                } catch (NumberFormatException ignored) {
                }
                try {
                    blue = Integer.parseInt(input.substring(4, 5), 16);
                } catch (NumberFormatException ignored) {
                }
                _disableEvents = true;
                seekBarRed.setProgress(red);
                seekBarGreen.setProgress(green);
                seekBarBlue.setProgress(blue);
                btnColor.setBackgroundColor(Color.argb(255, red, green, blue));
                txtColor.setText(toHex(red) + toHex(green) + toHex(blue));
                FireReceiver.setLight(red, green, blue, false);
                _disableEvents = false;
            }
        });
        txtDelay.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                if(_disableEvents) return;
                if(txtDelay.getText().toString().equals("0")) {
                    _disableEvents = true;
                    txtDelay.setText("");
                    _disableEvents = false;
                }
            }
        });
    }

    @Override
    public void finish()
    {
        FireReceiver.setLight(0, 0, 0, false);
        if (!isCanceled()) {
            try {
                final int red = seekBarRed.getProgress();
                final int green = seekBarGreen.getProgress();
                final int blue = seekBarBlue.getProgress();
                int delay = 0;
                Boolean off = false;
                if(_sync) {
                    try {
                        if(!txtDelay.getText().toString().equals("")) {
                            delay = Integer.parseInt(txtDelay.getText().toString());
                            if(delay < 0) delay = 0;
                        }
                    }
                    catch(NumberFormatException ignored) {
                    }
                    off = switchOff.isChecked();
                }

                if (red >= 0 && red <= 255 && green >= 0 && green <= 255 && blue >= 0 && blue <= 255) {
                    final Intent resultIntent = new Intent();

                    /*
                     * This extra is the data to ourselves: either for the Activity or the BroadcastReceiver. Note
                     * that anything placed in this Bundle must be available to Locale's class loader. So storing
                     * String, int, and other standard objects will work just fine. Parcelable objects are not
                     * acceptable, unless they also implement Serializable. Serializable objects must be standard
                     * Android platform objects (A Serializable class private to this plug-in's APK cannot be
                     * stored in the Bundle, as Locale's classloader will not recognize it).
                     */
                    final Bundle resultBundle =
                            PluginBundleManager.generateBundle(getApplicationContext(), red, green, blue, delay, off);
                    resultIntent.putExtra(com.twofortyfouram.locale.Intent.EXTRA_BUNDLE, resultBundle);

                    /*
                     * The blurb is concise status text to be displayed in the host's UI.
                     */
                    final String blurb = generateBlurb(getApplicationContext(), red, green, blue, delay, off);
                    resultIntent.putExtra(com.twofortyfouram.locale.Intent.EXTRA_STRING_BLURB, blurb);

                    if(_sync) {
                        TaskerPlugin.Setting.requestTimeoutMS(resultIntent, TaskerPlugin.Setting.REQUESTED_TIMEOUT_MS_NEVER);
                    }

                    setResult(RESULT_OK, resultIntent);
                }
            } catch (NumberFormatException ignored) {
            }
        }

        super.finish();
    }

    static String generateBlurb(final Context context, final int red, final int green, final int blue, final int delay, final Boolean off)
    {
        String message = "#" + toHex(red) + toHex(green) + toHex(blue);
        if(_sync && delay > 0) {
            message += " for " + delay + " ms";
            if(off) message += ", then turn off";
        }
        final int maxBlurbLength =
                context.getResources().getInteger(R.integer.twofortyfouram_locale_maximum_blurb_length);

        if (message.length() > maxBlurbLength)
        {
            return message.substring(0, maxBlurbLength);
        }

        return message;
    }

    public static String toHex(int input) {
        String hex = Integer.toHexString(input);
        StringBuilder sb = new StringBuilder();
        for(int toPrepend = 2 - hex.length(); toPrepend > 0; toPrepend--) {
            sb.append('0');
        }
        sb.append(hex);
        return sb.toString().toUpperCase();
    }

    public void btnSave_OnClick(View view) {
        mIsCancelled = false;
        finish();
    }

    public void btnCancel_OnClick(View view) {
        mIsCancelled = true;
        finish();
    }
}