/*
 * Copyright 2013 two forty four a.m. LLC <http://www.twofortyfouram.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * <http://www.apache.org/licenses/LICENSE-2.0>
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 */

package com.jfsoftware.ledcontrol;

import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

/**
 * Class for managing the {@link com.twofortyfouram.locale.Intent#EXTRA_BUNDLE} for this plug-in.
 */
public final class PluginBundleManager
{
    public static final String BUNDLE_EXTRA_INT_RED = "com.jfsoftware.ledcontrol.extra.INT_RED";
    public static final String BUNDLE_EXTRA_INT_GREEN = "com.jfsoftware.ledcontrol.extra.INT_GREEN";
    public static final String BUNDLE_EXTRA_INT_BLUE = "com.jfsoftware.ledcontrol.extra.INT_BLUE";
    public static final String BUNDLE_EXTRA_INT_DELAY = "com.jfsoftware.ledcontrol.extra.INT_DELAY";
    public static final String BUNDLE_EXTRA_BOOL_OFF = "com.jfsoftware.ledcontrol.extra.BOOL_OFF";

    /**
     * Type: {@code int}.
     * <p>
     * versionCode of the plug-in that saved the Bundle.
     */
    /*
     * This extra is not strictly required, however it makes backward and forward compatibility significantly
     * easier. For example, suppose a bug is found in how some version of the plug-in stored its Bundle. By
     * having the version, the plug-in can better detect when such bugs occur.
     */
    public static final String BUNDLE_EXTRA_INT_VERSION_CODE =
            "com.jfsoftware.ledcontrol.extra.INT_VERSION_CODE"; //$NON-NLS-1$

    /**
     * Method to verify the content of the bundle are correct.
     * <p>
     * This method will not mutate {@code bundle}.
     *
     * @param bundle bundle to verify. May be null, which will always return false.
     * @return true if the Bundle is valid, false if the bundle is invalid.
     */
    public static boolean isBundleValid(final Bundle bundle)
    {
        if (null == bundle)
        {
            return false;
        }

        /*
         * Make sure the expected extras exist
         */
        if (!bundle.containsKey(BUNDLE_EXTRA_INT_RED))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle must contain extra %s", BUNDLE_EXTRA_INT_RED));
            }
            return false;
        }
        if (!bundle.containsKey(BUNDLE_EXTRA_INT_GREEN))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle must contain extra %s", BUNDLE_EXTRA_INT_GREEN));
            }
            return false;
        }
        if (!bundle.containsKey(BUNDLE_EXTRA_INT_BLUE))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle must contain extra %s", BUNDLE_EXTRA_INT_BLUE));
            }
            return false;
        }
        if (!bundle.containsKey(BUNDLE_EXTRA_INT_DELAY))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle must contain extra %s", BUNDLE_EXTRA_INT_DELAY));
            }
            return false;
        }
        if (!bundle.containsKey(BUNDLE_EXTRA_BOOL_OFF))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle must contain extra %s", BUNDLE_EXTRA_BOOL_OFF));
            }
            return false;
        }
        if (!bundle.containsKey(BUNDLE_EXTRA_INT_VERSION_CODE))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                      String.format("bundle must contain extra %s", BUNDLE_EXTRA_INT_VERSION_CODE)); //$NON-NLS-1$
            }
            return false;
        }

        /*
         * Make sure the correct number of extras exist. Run this test after checking for specific Bundle
         * extras above so that the error message is more useful. (E.g. the caller will see what extras are
         * missing, rather than just a message that there is the wrong number).
         */
        if (6 != bundle.keySet().size())
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                      String.format("bundle must contain 5 keys, but currently contains %d keys: %s", bundle.keySet().size(), bundle.keySet())); //$NON-NLS-1$
            }
            return false;
        }

        int red = bundle.getInt(BUNDLE_EXTRA_INT_RED, -1);
        if(red < 0 || red > 255) {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle extra %s must be between 0 and 255", BUNDLE_EXTRA_INT_RED));
            }
            return false;
        }

        int green = bundle.getInt(BUNDLE_EXTRA_INT_GREEN, -1);
        if(green < 0 || green > 255) {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle extra %s must be between 0 and 255", BUNDLE_EXTRA_INT_GREEN));
            }
            return false;
        }

        int blue = bundle.getInt(BUNDLE_EXTRA_INT_BLUE, -1);
        if(blue < 0 || blue > 255) {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle extra %s must be between 0 and 255", BUNDLE_EXTRA_INT_BLUE));
            }
            return false;
        }

        int delay = bundle.getInt(BUNDLE_EXTRA_INT_DELAY, -1);
        if(delay < 0) {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                        String.format("bundle extra %s must be greater than or equal to 0", BUNDLE_EXTRA_INT_DELAY));
            }
            return false;
        }

        if (bundle.getInt(BUNDLE_EXTRA_INT_VERSION_CODE, 0) != bundle.getInt(BUNDLE_EXTRA_INT_VERSION_CODE, 1))
        {
            if (Constants.IS_LOGGABLE)
            {
                Log.e(Constants.LOG_TAG,
                      String.format("bundle extra %s appears to be the wrong type.  It must be an int", BUNDLE_EXTRA_INT_VERSION_CODE)); //$NON-NLS-1$
            }

            return false;
        }

        return true;
    }

    public static Bundle generateBundle(final Context context, final int red, final int green, final int blue, final int delay, final Boolean off)
    {
        final Bundle result = new Bundle();
        result.putInt(BUNDLE_EXTRA_INT_VERSION_CODE, Constants.getVersionCode(context));
        result.putInt(BUNDLE_EXTRA_INT_RED, red);
        result.putInt(BUNDLE_EXTRA_INT_GREEN, green);
        result.putInt(BUNDLE_EXTRA_INT_BLUE, blue);
        result.putInt(BUNDLE_EXTRA_INT_DELAY, delay);
        result.putBoolean(BUNDLE_EXTRA_BOOL_OFF, off);

        return result;
    }

    /**
     * Private constructor prevents instantiation
     *
     * @throws UnsupportedOperationException because this class cannot be instantiated.
     */
    private PluginBundleManager()
    {
        throw new UnsupportedOperationException("This class is non-instantiable"); //$NON-NLS-1$
    }
}